/** @file   objectstorage.h
 * @brief   Declaration of ObjectStorage - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_OBJECTSTORAGE_H
#define H_WWW_OBJECTSTORAGE_H

#include <vector>
#include <string>
#include "player.h"


namespace WeWantWar {

/** @class  ObjectStorage
 * @brief   This class manages the instances of GameObject.
 * @author  Tomi Lamminsaari
 *
 * All the "living" creatures (descendant of GameObject) are held in an instance
 * of this class. These creatures are owned by this object so when you delete
 * the ObjectStorage - instance, it deletes all the GameObjects it contains.
 *
 * There are some methods that provide the access to the GameObjects but the
 * object tables are public members just to give you free access to do what
 * ever you want to do with the objects.
 */
class ObjectStorage
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Every GameObject in the game are stored in this vector. */
  std::vector< GameObject* > objectList;
  
  /** The killed objects are placed in this vector. */
  std::vector< GameObject* > killedObjects;
  /** Here are the civilians. */
  std::vector< GameObject* > civilianObjects;
  /** All the cars and other vehicles are in this vector. */
  std::vector< GameObject* > vehicleObjects;
  /** All the hostile enemies are in this vector. */
  std::vector< GameObject* > hostileObjects;

  
  /** The player is stored in this member. Player is also the first element
   * in @c objectList - vector.
   */
  Player* pPlayer;
  
  /** This member holds the primary target object for the AI-objects. By
   * default this should be the same pointer as in pPlayer - member.
   */
  GameObject* pPrimaryTarget;
  
  /** This vector holds the list of the gameobjects that can be the targets.
   * If primary target is nullpointer, then the ai-controllers pick randomly
   * one of the objects stored here to be the target of the objects.
   */
  std::vector< GameObject* > secondaryTargets;
  
  

  /** Datatype for the multimap that stores the pointers by objects ID-codes */
  typedef std::multimap< GameObject::IDCode, GameObject* > IDTable;
  
  /** This map holds the objects arranged by their id-codes. */
  std::multimap< GameObject::IDCode, GameObject* > objectByID;
  
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	ObjectStorage();


	/** Destructor
   */
	virtual ~ObjectStorage();

private:

	/** Copy constructor.
   * @param     rO                Reference to another ObjectStorage
   */
  ObjectStorage( const ObjectStorage& rO );

	/** Assignment operator
   * @param     rO                Reference to another ObjectStorage
   * @return    Reference to us.
   */
  ObjectStorage& operator = ( const ObjectStorage& rO );

public:


  ///
  /// Methods
  /// =======

  /** Adds new GameObject to this list.
   * @param     pO                Pointer to the object being added.
   */
  void addObject( GameObject* pO );
  
  /** Removes the given object from the tables.
   * @param     pO                Pointer to object being removed.
   */
  void remObject( GameObject* pO );
  
  /** Removes the object from given position.
   * @param     pos               Position of the object in objectList
   */
  void remObject( int pos );
  
  /** Updates all the objects.
   */
  void updateObjects();
  
  /** Redraws all the objects
   * @param     pQueue            Pointer to RedrawQueue - object
   */
  void redrawObjects( RedrawQueue* pQueue );
  
  /** Removes all the objects.
   */
  void clear();
  


  ///
  /// Getter methods
  /// ==============

  /** Tells if there are any object with given id-code still alive.
   * @param     id                ID of the object.
   * @return    <code>true</code> if there are objects with given id.
   */
  bool hasObjectWithID( GameObject::IDCode id ) const;

  /** Returns the pointers to the objects that have certain ID-code.
   * @param     id                ID of the objects
   * @param     rV                A vector where the pointers will be pushed.
   * @return    <code>true</code> if there were at least one object with
   *            given ID-code.
   */
  bool getObjectsWithID( GameObject::IDCode id, std::vector<GameObject*>& rV ) const;
  
  /** Tells if there are gameobjects inside the given rectangle. Both living
   * and killed objects are taken into account.
   * @param     rRect             The rectangle.
   * @return    <code>true</code> if there are anyone inside the rectangle.
   */
  bool objectsInsideRect( const eng2d::Rect2D& rRect ) const;
  
  /** Returns the first found object that has the given id.
   * @param   aId               Id of the object.
   * @return  Pointer to object that has the id or nullpointer if none were
   *          found.
   */
  GameObject* findObject( GameObject::IDCode aId ) const;

protected:

  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: objectstorage.h,v $
 * Revision 1.2  2006/05/15 19:21:28  lamminsa
 * findObject() method added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:08+02  lamminsa
 * Initial revision
 *
 */
 
